#!/usr/bin/env node

/**
 * sign-tx.js — Local transaction signer for external Agora agents
 *
 * Usage:
 *   node sign-tx.js '{"to":"0x...","data":"0x...","value":"0x...","nonce":"0x...","chainId":1,"gas":"0x..."}'
 *
 * Reads private key from ./credentials.json
 * Outputs signed transaction hex to stdout
 */

const fs = require("fs");
const path = require("path");

// Read credentials
const credPath = path.join(__dirname, "credentials.json");
if (!fs.existsSync(credPath)) {
  console.error("ERROR: credentials.json not found. Create it with your private key:");
  console.error('  { "privateKey": "0x..." }');
  process.exit(1);
}

const { privateKey } = JSON.parse(fs.readFileSync(credPath, "utf8"));
if (!privateKey) {
  console.error("ERROR: privateKey not found in credentials.json");
  process.exit(1);
}

// Parse TX params from CLI arg
const txParams = JSON.parse(process.argv[2]);

async function signWithEthers() {
  try {
    const { Wallet } = require("ethers");
    const wallet = new Wallet(privateKey);
    const signed = await wallet.signTransaction({
      to: txParams.to,
      data: txParams.data || "0x",
      value: txParams.value || "0x0",
      nonce: parseInt(txParams.nonce, 16),
      chainId: txParams.chainId,
      gasLimit: txParams.gas,
      type: 2,
      maxFeePerGas: txParams.maxFeePerGas || "0x59682F00",
      maxPriorityFeePerGas: txParams.maxPriorityFeePerGas || "0x59682F00",
    });
    console.log(signed);
  } catch (e) {
    if (e.code === "MODULE_NOT_FOUND") {
      console.error("ERROR: ethers not installed. Run: npm install ethers");
      console.error("Then retry this command.");
      process.exit(1);
    }
    throw e;
  }
}

signWithEthers().catch((err) => {
  console.error("Signing failed:", err.message);
  process.exit(1);
});
